### 模式切换Q&A

在实时系统开发中，模式切换（Mode Switch）是一个重要的概念，它涉及到线程从实时模式切换到非实时模式的过程。这种切换可能会引入不可预测的延迟，影响系统的实时性能。本文将针对开发者在实践中常遇到的几个问题，详细解释读写文件和使用 `printf` 是否会导致模式切换，以及如何在实时系统中正确处理这些操作。

---
### 读写文件会不会造成模式切换？

#### 读取文件

问题：在实时线程中，读取文件是否会导致模式切换？

回答：是的，读取文件会导致模式切换。

**原因分析**

在实时线程中，读取文件通常涉及以下操作：

- 磁盘 I/O：读取文件需要访问磁盘，而磁盘 I/O 操作的延迟不可预测，可能会阻塞线程。
- 系统调用：调用诸如 `open`、`read`、`lseek` 等标准文件操作函数，这些函数属于 `Linux` 的系统调用，会导致线程从实时模式切换到非实时模式。

在 `xkernel` 的 `Cobalt` 实时内核中，实时线程调用非实时的系统调用会被自动切换到非实时模式，以避免阻塞实时调度器。这种模式切换会引入额外的延迟，破坏实时性的要求。

**解决方案**

为了避免模式切换，开发者可以采取以下方法：

**使用 mmap 映射文件**

`mmap` 函数可以将文件映射到进程的地址空间，可以将文件内容预先加载并锁定到内存中，避免实时线程在运行时发生磁盘 I/O 操作。

**实现步骤**：

1. 使用 mmap 映射文件:

```c
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>

int fd = open("data_file.txt", O_RDONLY);
if (fd < 0) {
    perror("open failed");
    exit(EXIT_FAILURE);
}

struct stat sb;
if (fstat(fd, &sb) == -1) {
    perror("fstat failed");
    exit(EXIT_FAILURE);
}

size_t file_size = sb.st_size;

void *file_data = mmap(NULL, file_size, PROT_READ, MAP_PRIVATE, fd, 0);
if (file_data == MAP_FAILED) {
    perror("mmap failed");
    exit(EXIT_FAILURE);
}

close(fd);
```

2. 预读取文件内容：

为了确保文件内容实际加载到内存中，可以在初始化阶段访问映射的内存区域：

```c
volatile char temp;
for (size_t i = 0; i < file_size; i += sysconf(_SC_PAGESIZE)) {
    temp = ((char *)file_data)[i];
}
```

作用：通过逐页访问，触发页面的实际加载。

3. 在实时线程中访问文件内容：

- 由于文件内容已经在内存中，实时线程可以直接访问 file_data，无需调用任何可能导致模式切换的系统调用。

**优点**：

- 避免在实时线程中发生模式切换。
- 提高文件访问的效率和响应速度。

**注意事项**：

- 内存占用：大型文件映射到内存可能占用大量内存，需要确保系统有足够的内存资源。
- 文件更新：如果文件内容在运行期间需要更新，需要重新映射或采用其他机制同步。

**将文件读取操作移到非实时线程**

另一种方法是将文件读取操作放在非实时线程或进程中，实时线程通过共享内存、消息队列等方式与非实时线程通信，获取所需的数据。

**实现步骤**：

**步骤 1 创建非实时线程**

```c
pthread_t non_rt_thread;
pthread_create(&non_rt_thread, NULL, non_rt_thread_function, NULL);
```

**步骤 2 在非实时线程中读取文件并加载数据**

```c
void *non_rt_thread_function(void *arg) {
    // 读取文件并处理数据
    // 将数据传递给实时线程
    return NULL;
}
```

**步骤 3 实时线程从共享内存或消息队列中获取数据**

```c
void *rt_thread_function(void *arg) {
    // 从共享内存或消息队列中获取数据
    return NULL;
}
```


**优点**：

- 避免实时线程执行阻塞的 I/O 操作。
- 实时线程专注于处理时间敏感的任务。

#### 写入文件

问题：在实时线程中，写入文件是否会导致模式切换？

回答：不会，写入文件在一定条件下不会导致模式切换。

**原因分析**

在 `xkernel` 的 `Cobalt` 实时内核中，标准的文件写入操作通常会导致模式切换。`xkernel` 提供了特殊的机制，使实时线程可以在不发生模式切换的情况下进行日志记录和数据写入。

- 实时核缓冲区：实时线程的输出（如 `printf`）会被重定向到实时核的内存缓冲区。
- 后台处理：非实时核会在适当的时候将缓冲区内容写入实际的文件或终端。

**具体机制**

- rt_print_auto_init() 函数：在程序初始化时调用该函数，设置实时线程的输出自动初始化。

```c
#include <rtdk.h>

int main() {
    rt_print_auto_init(1);
    // 其他初始化代码
}
```

- 实时线程中使用 `rt_printf`：在实时线程中使用 `rt_printf` 代替标准的 `printf`，不会导致模式切换。或者使用`--wrap`方式替换了posix调用。

```c
void *rt_thread_function(void *arg) {
    rt_printf("Real-time thread output\n");
    return NULL;
}
```

**注意事项**

- 缓冲区大小：实时核的内存缓冲区是有限的，需要确保不会溢出。
- 数据丢失风险：如果系统崩溃或异常退出，缓冲区中的数据可能未被写入文件，导致数据丢失。

**适用场景**

- 日志记录：实时线程需要记录日志信息，但又不能发生模式切换。
- 数据采集：实时线程需要将数据保存到文件，但不能阻塞或引入延迟。

#### printf 会不会造成模式切换？

问题：在实时线程中，使用 `printf` 函数是否会导致模式切换？

回答：与写文件同理，使用 `printf` 不会导致模式切换，但需要注意使用正确的函数和方法。

**原因分析**

标准的 `printf` 函数在内部可能会调用系统调用，如 `write`，这些系统调用会导致模式切换。`xkernel` 提供了实时安全的输出函数，可以避免模式切换。

**注意事项**：

- 缓冲区溢出：需要确保输出量不会超过缓冲区容量（默认为16k, 可通过参数放大，具体参照[标准命令行选项](https://help.sinsegye.com.cn/docs/SinsegyeMetaOSoptions)）。

---
### 使用 `mlockall()` 后，进程的所有内存页面都会被锁定，防止被换出到交换空间。那么，为什么在循环任务中仍然要避免使用 malloc 函数呢？

**原因分析**：

1. 非确定性的内存分配时间：
    - `malloc` 的执行时间不可预测：标准的 `malloc` 实现可能涉及复杂的内存管理算法，包括查找空闲块、合并碎片、更新内部数据结构等。这些操作的执行时间取决于内存的当前状态，可能会导致不可预测的延迟。
    - 实时系统要求确定性：在硬实时系统中，任务需要在严格的时间约束内完成。任何不可预测的延迟都可能导致任务超时，影响系统的实时性能。
2. 可能触发缺页异常（Page Fault）：
    - `mlockall()` 只能锁定已分配的物理内存页面：`mlockall(MCL_CURRENT | MCL_FUTURE) `会锁定当前已映射到物理内存的页面，以及未来分配时自动锁定内存页面。
    - `malloc` 分配新的内存页面：当调用 `malloc` 请求新的内存块时，如果当前的内存映射区域不足，操作系统需要分配新的物理内存页面并映射到进程的地址空间。这可能导致缺页异常，触发内核分配内存页面的过程。
    - 缺页异常导致不可预测的延迟：缺页异常处理涉及内核态的内存管理操作，如查找空闲物理内存、更新页表等。这些操作的执行时间不确定，可能会导致任务的执行时间超出预期。
3. 内部锁竞争和线程安全问题：
    - `malloc` 的线程安全实现：为了支持多线程，标准的 `malloc` 实现通常需要使用锁（如互斥锁）来保护内部数据结构。在高并发的环境下，多个线程同时调用 `malloc`，可能导致锁竞争，增加了执行时间的不确定性。
    - 锁竞争带来的延迟：当一个线程持有锁时，其他线程需要等待锁释放，可能导致不可预测的等待时间。这对于实时任务来说是不可接受的。

**解决方案**
- 预先分配内存：在任务初始化时，预先分配所需的内存，避免在任务运行过程中进行动态内存分配。
- 使用实时内存分配器：如果确实需要动态内存分配，可以使用专门为实时系统设计的内存分配器，如 `xkernel` 提供的 `rt_heap`，或其他具有确定性行为的内存池机制。